/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: streamline.c,v 1.52.2.5 2003/03/05 17:02:47 sasa Exp $
 *
 * Author  : SaSa
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/stream.h>
#include <zorp/streamline.h>
#include <zorp/log.h>

#include <glib.h>

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <assert.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#else
#  include <sys/socket.h>
#  include <sys/poll.h>
#endif

#define ZRL_IGNORE_TILL_EOL     0x00010000

typedef struct _ZStreamLine
{
  ZStream super;

  guint flags;
  gchar *buffer;
  gsize bufsize, pos, end, oldpos;
  GIOStatus last_state;
            
} ZStreamLine;

static inline gboolean
z_stream_line_have_line(ZStreamLine *self)
{
  guint avail = self->end - self->pos;
  char *eol = memchr(self->buffer + self->pos, '\n', avail);
  gboolean ret = FALSE;

  z_enter();
  
  if (eol != NULL)
    {
      ret = TRUE;
    }

  z_leave();
  return ret;
}

static GIOStatus
z_stream_line_read_method(ZStream  *stream,
                            gchar  *buf,
                            gsize   count,
                            gsize  *bytes_read,
                           GError **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;
  guint avail = self->end - self->pos;
  
  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);
  
  if (avail)
    {
      *bytes_read = MIN(count, avail);
      memmove(buf, self->buffer + self->pos, *bytes_read);
      
      self->oldpos = self->pos;
      self->pos += *bytes_read;

      if (self->pos == self->end)
        self->pos = self->end = 0;

      res = G_IO_STATUS_NORMAL;
    }
  else
    {
      if (self->last_state == G_IO_STATUS_NORMAL)
        {
          self->super.parent->timeout = self->super.timeout;
          res = z_stream_read(self->super.parent, buf, count, bytes_read, error);
        }
      else
        res = self->last_state;
    }
  
  z_leave();
  return res;
}

static GIOStatus
z_stream_line_write_method(ZStream  *stream,
                       const gchar  *buf,
                             gsize   count,
                             gsize  *bytes_written,
                            GError **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;

  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  self->super.parent->timeout = self->super.timeout;
  res = z_stream_write(self->super.parent, buf, count, bytes_written, error);
  
  z_leave();
  return res;
}

static GIOStatus
z_stream_line_write_pri_method(ZStream  *stream,
                           const gchar  *buf,
                                 gsize   count,
                                 gsize  *bytes_written,
                                GError **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;
  
  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);
  
  self->super.parent->timeout = self->super.timeout;
  res = z_stream_write_pri(self->super.parent, buf, count, bytes_written, error);
  
  z_leave();
  return res;
}

static GIOStatus
z_stream_line_shutdown_method(ZStream *stream, int i, GError **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;
  
  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  res = z_stream_shutdown(self->super.parent, i, error);
  
  z_leave();
  return res;

}

static GIOStatus
z_stream_line_close_method(ZStream *stream, GError **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;

  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  res = z_stream_close(self->super.parent, error);
  
  z_leave();
  return res;
}

static gboolean
z_stream_line_read_callback(ZStream *stream, GIOCondition poll_cond, gpointer s)
{
  ZStreamLine *self = (ZStreamLine *) s;
  guint avail = self->bufsize - self->end;
  GIOStatus rc;
  gsize bytes_read;

  z_enter();
  
  if (avail)
    {
      self->super.parent->timeout = self->super.timeout;
      rc = z_stream_read(self->super.parent, self->buffer + self->end, avail, &bytes_read, NULL);

      switch (rc)
        {
        case G_IO_STATUS_NORMAL:
          self->end += bytes_read;
        case G_IO_STATUS_AGAIN:
          z_leave();
          return TRUE;
        case G_IO_STATUS_EOF:
        default:
          self->last_state = rc;
          z_leave();
          return FALSE;
        }
    }
  else if (!z_stream_line_have_line(self))
    {
      rc = self->super.read_cb(s, G_IO_IN, self->super.user_data_read);
    }

  z_leave();
  return TRUE;
}

static gboolean
z_stream_line_write_callback(ZStream *stream, GIOCondition poll_cond, gpointer s)
{
  ZStreamLine *self = (ZStreamLine *) s;
  gboolean rc;

  z_enter();
  
  rc = (*self->super.write_cb)(s, poll_cond, self->super.user_data_write);
  
  z_leave();
  return rc;
}

static gboolean
z_stream_line_pri_callback(ZStream *stream, GIOCondition poll_cond, gpointer s)
{
  ZStreamLine *self = (ZStreamLine *) s;
  gboolean rc;

  z_enter();
  
  rc = (*self->super.pri_cb)(s, poll_cond, self->super.user_data_pri);
  
  z_leave();
  return rc;
}

static gboolean
z_stream_line_ctrl_method(ZStream *s, guint function, gpointer value, guint vlen)
{
  ZStreamLine *self = (ZStreamLine *)s;
  gboolean ret = FALSE;
  
  z_enter();
  assert(s->type == ZST_LINE);
  
  switch (ZST_CTRL_MSG(function))
    {
    case ZST_LINE_SET_TRUNCATE:
      if (vlen == sizeof(gboolean))
        {
          gboolean flag = *((gboolean *)value);
          if (flag)
            self->flags |= ZRL_TRUNCATE;
          else
            self->flags &= ~ZRL_TRUNCATE;
          
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_SET_SPLIT:
      if (vlen == sizeof(gboolean))
        {
          gboolean flag = *((gboolean *)value);
          if (flag)
            self->flags |= ZRL_SPLIT;
          else
            self->flags &= ~ZRL_SPLIT;
          
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_SET_SINGLE_READ:
      if (vlen == sizeof(gboolean))
        {
          gboolean flag = *((gboolean *)value);
          if (flag)
            self->flags |= ZRL_SINGLE_READ;
          else
            self->flags &= ~ZRL_SINGLE_READ;
          
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_SET_PARTIAL_READ:
      if (vlen == sizeof(gboolean))
        {
          gboolean flag = *((gboolean *)value);
          if (flag)
            self->flags |= ZRL_PARTIAL_READ;
          else
            self->flags &= ~ZRL_PARTIAL_READ;
          
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_GET_TRUNCATE:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = !!(self->flags & ZRL_TRUNCATE);
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_GET_SPLIT:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = !!(self->flags & ZRL_SPLIT);
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_GET_SINGLE_READ:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = !!(self->flags & ZRL_SINGLE_READ);
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_GET_PARTIAL_READ:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = !!(self->flags & ZRL_PARTIAL_READ);
          z_leave();
          return TRUE;
        }
      break;
    case ZST_CTRL_SET_CALLBACK_READ:
    case ZST_CTRL_SET_CALLBACK_WRITE:
    case ZST_CTRL_SET_CALLBACK_PRI:
      ret = z_stream_ctrl_method(s, function, value, vlen);
      break;
    default:
      ret = z_stream_ctrl_method(s, ZST_CTRL_MSG_FORWARD | function, value, vlen);
      break;
    }
  z_leave();
  return ret;
}

/*+ function to fetch a line from the buffer +*/
static GIOStatus
z_stream_line_get_from_buf(ZStreamLine *self, char **line, guint *length)
{
  guint avail = self->end - self->pos;
  char *eol = memchr(self->buffer + self->pos, '\n', avail);
  char *nul;

  z_enter();

  if (eol)
    {
      *length = eol - (self->buffer + self->pos) + 1;
      *line = self->buffer + self->pos;
      self->oldpos = self->pos;
      self->pos += *length;

      nul = memchr(*line, '\0', *length);
      if (nul)
        {
          if (self->flags & ZRL_NUL_NONFATAL)
            {
              *length -= eol - nul;
              z_leave();
              return G_IO_STATUS_NORMAL;
            }
          z_leave();
          return G_IO_STATUS_ERROR;
        }

      if (self->flags & ZRL_EOL_NL)
        {
          (*length)--;
        }
      else if (self->flags & ZRL_EOL_CRLF)
        {
          (*length)--;
          if (eol - self->buffer >= 1 && *(eol - 1) == '\r')
            {
              (*length)--;
            }
          else if (self->flags & ZRL_EOL_FATAL)
            {
              z_leave();
              return G_IO_STATUS_ERROR;
            }
        }
      z_leave();
      return G_IO_STATUS_NORMAL;
    }
  else if (self->pos)
    {
      *length = 0;
      memmove(self->buffer, self->buffer + self->pos, avail);
      self->end = avail;
      self->pos = 0;
      self->oldpos = 0;
    }
  z_leave();
  return G_IO_STATUS_AGAIN;
}

GIOStatus
z_stream_line_get(ZStream *stream, char **line, guint *length, GError **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  gsize avail, bytes_read;
  gint rc;

  z_enter();
  
  while (self && z_stream_get_type((ZStream *) self) != ZST_LINE)
    self = (ZStreamLine *) self->super.parent;
  
  if (!self)
    {
      /*LOG
        This message indicate by an internal error
       */
      z_log(NULL, CORE_ERROR, 2, "Internal error; reason='Bad stream type'");
      z_leave();
      return G_IO_STATUS_ERROR;
    }
  
  if (self->end != self->pos)
    {
      /* we have something, try to return it */
      rc = z_stream_line_get_from_buf(self, line, length);
      if (rc == G_IO_STATUS_NORMAL)
        {
          z_leave();
          return G_IO_STATUS_NORMAL;
        }
      /* the available data is now at the beginning of our buffer */
    }
  else
    {
      self->pos = self->end = self->oldpos = 0;
    }

  *length = 0;
  *line = NULL;

  if (self->last_state != G_IO_STATUS_NORMAL)
    {
      z_leave();
      return self->last_state;
    }

  while (1)
    {
      avail = self->bufsize - self->end;
      if (!avail)
        {
          /*
           * this means that there's no space in the buffer, and no eol could
           * be found
           */
          if (self->flags & ZRL_IGNORE_TILL_EOL)
            {
              self->pos = self->end = self->oldpos = 0;
              avail = self->bufsize;
            }
          else if (self->flags & ZRL_TRUNCATE)
            {
              *line = self->buffer;
              *length = self->bufsize;
              self->pos = self->end = self->oldpos = 0;
              self->flags |= ZRL_IGNORE_TILL_EOL;
              z_leave();
              return G_IO_STATUS_NORMAL;
            }
          else if (self->flags & ZRL_SPLIT)
            {
              *line = self->buffer;
              *length = self->bufsize;
              self->pos = self->end = self->oldpos = 0;
              z_leave();
              return G_IO_STATUS_AGAIN;
            }
          else
            {
              /*LOG
                This message is sent when the proxy reading too long
                input line. This may cause by a cracking attempt. But if
                not try to increase max_line_length.
               */
              z_log(NULL, CORE_ERROR, 2, "Line too long; buffer='%.*s'", (gint) self->bufsize, self->buffer);
              z_leave();
              *line = NULL;
              *length = 0;
              return G_IO_STATUS_ERROR;
            }
        }

      self->super.parent->timeout = self->super.timeout;
      rc = z_stream_read(self->super.parent, self->buffer + self->end, avail, &bytes_read, error);
      switch (rc)
        {
        case G_IO_STATUS_NORMAL:
          self->end += bytes_read;
          rc = z_stream_line_get_from_buf(self, line, length);
          switch (rc)
            {
            case G_IO_STATUS_NORMAL:
              if (self->flags & ZRL_IGNORE_TILL_EOL)
                {
                  self->flags &= ~ZRL_IGNORE_TILL_EOL;
                  rc = G_IO_STATUS_AGAIN;
                  /* No break, go to next case */
                }
              else
                {
                  z_leave();
                  return rc;
                }
            case G_IO_STATUS_AGAIN:
              if (self->flags & ZRL_SINGLE_READ)
                {
                  *line = NULL;
                  *length = 0;
                  z_leave();
                  return rc;
                }
              break;
            default:
              *line = NULL;
              *length = 0;
              z_leave();
              return rc;
            }
          break;
        case G_IO_STATUS_EOF:
          z_leave();
          return G_IO_STATUS_EOF;
        case G_IO_STATUS_AGAIN:
          *line = NULL;
          *length = 0;
          z_leave();
          return G_IO_STATUS_AGAIN;
        default:
          z_leave();
          return G_IO_STATUS_ERROR;
        }
    }
}

GIOStatus
z_stream_line_get_copy(ZStream *s, char *line, guint *length, GError **error)
{
  gchar *b;
  guint len;
  GIOStatus res;
  
  z_enter();
  
  res = z_stream_line_get(s, &b, &len, error);
  if (res == G_IO_STATUS_NORMAL || (res == G_IO_STATUS_AGAIN && len > 0))
    {
      if (len > *length)
        {
          z_leave();
          return G_IO_STATUS_ERROR;
        }
      else
        {
          *length = len;
          memcpy(line, b, len);
          z_leave();
          return res;
        }
    }
  else
    {
      *length = 0;
    }
  z_leave();
  return res;
}

void
z_stream_line_unget_line(ZStream *stream)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  
  z_enter();
  
  while (self && z_stream_get_type((ZStream *) self) != ZST_LINE)
    self = (ZStreamLine *) self->super.parent;
  
  if (!self)
    {
      /*LOG
        This message indicate by an internal error
       */
      z_log(NULL, CORE_ERROR, 3, "Internal error; reason='Bad stream type'");
      z_leave();
      return;
    }
  
  self->pos = self->oldpos;
  
  z_leave();
}


static void
z_stream_line_attach_source_method(ZStream *stream, GMainContext *context)
{
  ZStreamLine *self = (ZStreamLine *)stream;

  z_enter();

  z_stream_ref(stream);
  z_stream_attach_source(self->super.parent, context);
  if (!stream->source)
    {
      stream->source = z_stream_source_new(stream);
      g_source_attach(stream->source, context);
    }
  z_stream_unref(stream);
  z_leave();
}

static void
z_stream_line_detach_source_method(ZStream *stream)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GSource *source;
  
  z_enter();
  if (stream->source)
    {
      source = stream->source;
      stream->source = NULL;
      /*
         NOTE Must be in this order because
         g_source_unref may drop the last
         reference to source.
       */
      g_source_destroy(source);
      g_source_unref(source);
    }

  z_stream_detach_source(self->super.parent);
    
  z_leave();
}

static void
z_stream_line_free(ZStream *stream)
{
  ZStreamLine *self = (ZStreamLine *) stream;

  z_enter();
  
  z_stream_unref(self->super.parent);
  g_free(self->buffer);
  g_free(self);
  
  z_leave();
}


static gboolean z_stream_line_watch_prepare(ZStream *s,
                                            GSource *src,
                                               gint *timeout);

static gboolean z_stream_line_watch_check(ZStream *s, GSource *src);

static gboolean z_stream_line_watch_dispatch (ZStream *s,
                                              GSource *src);

ZStreamFuncs z_stream_line_funcs =
{
  z_stream_line_read_method,
  z_stream_line_write_method,
  NULL,
  z_stream_line_write_pri_method,
  z_stream_line_shutdown_method,
  z_stream_line_close_method,
  z_stream_line_ctrl_method,
  
  z_stream_line_attach_source_method,
  z_stream_line_detach_source_method,
  z_stream_line_watch_prepare,
  z_stream_line_watch_check,
  z_stream_line_watch_dispatch,
  NULL,
  z_stream_line_free
};

ZStream *
z_stream_line_new(ZStream *stream, guint bufsize, guint flags)
{
  ZStreamLine *self = g_new0(ZStreamLine, 1);

  z_enter();

  z_stream_init(&self->super, ZST_LINE, &z_stream_line_funcs, stream->name);
  self->super.parent = stream;
  self->flags = flags;
  self->bufsize = bufsize;
  self->buffer = g_new(gchar, bufsize);
  self->super.timeout = self->super.parent->timeout;
  self->last_state = G_IO_STATUS_NORMAL;
  
  z_stream_ref(self->super.parent);
  
  z_stream_set_callback(self->super.parent, Z_STREAM_FLAG_READ, z_stream_line_read_callback, self, NULL);
  z_stream_set_callback(self->super.parent, Z_STREAM_FLAG_WRITE, z_stream_line_write_callback, self, NULL);
  z_stream_set_callback(self->super.parent, Z_STREAM_FLAG_PRI, z_stream_line_pri_callback, self, NULL);

  z_leave();
  return (ZStream *) self;
}

static gboolean
z_stream_line_may_dispatch(ZStreamLine *self)
{
  z_enter();
  if (self->last_state != G_IO_STATUS_NORMAL)
    {
      z_leave();
      return TRUE;
    }

  if (self->flags & ZRL_PARTIAL_READ)
    {
      z_leave();
      return !!(self->end - self->pos);
    }

  if (z_stream_line_have_line(self))
    {
      z_leave();
      return TRUE;
    }

  z_leave();
  return FALSE;
}

static gboolean 
z_stream_line_watch_prepare(ZStream *s, GSource *src, gint *timeout)
{
  ZStreamLine *self = (ZStreamLine *) s;
  gboolean ret = FALSE;

  z_enter();
  
  *timeout = -1;

  if (s->want_read)
    {
      ret = z_stream_line_may_dispatch(self);
    }
  z_leave();
  return ret;
}

static gboolean 
z_stream_line_watch_check(ZStream *s, GSource *src)
{
  ZStreamLine *self = (ZStreamLine *) s;
  gboolean ret = FALSE;

  z_enter();

  if (s->want_read)
    {
      ret = z_stream_line_may_dispatch(self);
    }

  z_leave();
  return ret;
}

static
gboolean 
z_stream_line_watch_dispatch(ZStream *s,
                             GSource *src)
{
  ZStreamLine *self = (ZStreamLine *) s;
  gboolean rc;

  z_enter();

  rc = self->super.read_cb(s, G_IO_IN, self->super.user_data_read);

  z_leave();
  return rc;
}

